/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: SCF Card object sample. This sample does not require any CardService.
 *
 */

package cardobject;
import jp.co.ricoh.dsdk.panel.AttributeType;
import jp.co.ricoh.dsdk.panel.Button;
import jp.co.ricoh.dsdk.panel.Container;
import jp.co.ricoh.dsdk.panel.Font;
import jp.co.ricoh.dsdk.panel.Frame;
import jp.co.ricoh.dsdk.panel.LED;
import jp.co.ricoh.dsdk.panel.LEDExecutionException;
import jp.co.ricoh.dsdk.panel.Label;
import jp.co.ricoh.dsdk.panel.Language;
import jp.co.ricoh.dsdk.panel.SoftKeyboard;
import jp.co.ricoh.dsdk.panel.SoftKeyboardOpenException;
import jp.co.ricoh.dsdk.panel.Text;
import jp.co.ricoh.dsdk.panel.Window;
import jp.co.ricoh.dsdk.panel.LED.Operation;
import jp.co.ricoh.dsdk.panel.LED.Type;
import jp.co.ricoh.dsdk.panel.event.ActionEvent;
import jp.co.ricoh.dsdk.panel.event.ActionListener;
import jp.co.ricoh.dsdk.panel.event.KeyAdapter;
import jp.co.ricoh.dsdk.panel.event.KeyEvent;
import jp.co.ricoh.dsdk.scard.framework.card.Card;
import jp.co.ricoh.dsdk.scard.framework.card.CardAccessException;
import jp.co.ricoh.dsdk.scard.framework.card.CardManager;
import jp.co.ricoh.dsdk.scard.framework.card.InitializeException;
import jp.co.ricoh.dsdk.scard.framework.card.Slot;
import jp.co.ricoh.dsdk.scard.framework.card.SlotAccessException;
import jp.co.ricoh.dsdk.scard.framework.card.Card.IO;
import jp.co.ricoh.dsdk.scard.framework.card.Card.Info;
import jp.co.ricoh.dsdk.scard.framework.event.CardEvent;
import jp.co.ricoh.dsdk.scard.framework.event.CardEventListener;
import jp.co.ricoh.dsdk.scard.framework.util.APDU;
import jp.co.ricoh.dsdk.xlet.UnavailableContainerException;
import jp.co.ricoh.dsdk.xlet.Xlet;
import jp.co.ricoh.dsdk.xlet.XletContext;
import jp.co.ricoh.dsdk.xlet.XletStateChangeException;

/**
 * ScfCardObject
 * This is a sample application using the Card object of the SmartCard Framework.
 * 
 * ScfCardObject
 * SmartCard FrameworkのCardオブジェクトを利用するサンプルプログラムです。
 */
public class ScfCardObject implements Xlet, ActionListener {
	
	/**
	 * The CardManager object.
	 * 
	 * CardManagerオブジェクト 
	 */
	private CardManager cardManager;
	
	/**
	 * The CardEventListener object.
	 * 
	 * CardEventListenerオブジェクト
	 */
	private CardEventListener listener;
	
	/**
	 * The byte array that stores APDU request.
	 * 
	 * APDUリクエストを格納するバイト配列
	 */
	private byte[] request;

	/**
	 * The root frame.
	 * 
	 * ルートフレーム
	 */
	private Frame frame;
	
	/**
	 * The label to be used to display a message.
	 * 
	 * メッセージ表示ラベル
	 */
	private Label msgLabel;
	
	/**
	 * The label to be used to display a slot information.
	 * 
	 * スロット表示ラベル
	 */
	private Label slotLabel;
	
	/**
	 * The label to be used to display a protocol information.
	 * 
	 * プロトコル表示ラベル
	 */
	private Label protocolLabel;
	
	/**
	 * The label to be used to display an ATR information.
	 * 
	 * ATR表示ラベル
	 */
	private Label atrLabel;
	
	/**
	 * The label to be used to display an APDU request.
	 * 
	 * リクエスト表示ラベル
	 */
	private Label requestLabel;
	
	/**
	 * The label to be used to display an APDU response.
	 * 
	 * レスポンス表示ラベル
	 */
	private Label responseLabel;
	
	/**
	 * The soft keyboard to be used to input APDU request.
	 * 
	 * カードに送信するAPDUを入力するためのソフトキーボード。
	 */
	private SoftKeyboard keyboard;
	
	/**
	 * The button to be used to display the soft keyboard.
	 * 
	 * ソフトキーボードを表示するボタン。
	 */
	private Button keyButton;
	
	/**
	 * This is an initialization process.
	 * 1.obtains the root frame.
	 * 2.creates GUI.
	 * 3.creates a CardManager object.
	 * 4.creates CardEventListener and adds it to the CardManager.
	 * 5.sets LED state.
	 *  
	 * 初期化処理です。
	 * 1.ルートフレームを取得する。
	 * 2.GUIを作成する。
	 * 3.CardManagerオブジェクトを生成する。
	 * 4.リスナを生成し、カードマネージャにセットする。
	 * 5.LEDを設定する。
	 */
	synchronized public void initXlet(XletContext xletContext) throws XletStateChangeException {	
		try {
			Container parent = xletContext.getContainer();
			while (!(parent instanceof Frame)) {
				parent = parent.getParent();
				if (parent == null) break;
			}
			if (parent == null) {
				return;
			}
			frame = (Frame) parent;

			createGUI();
			
		} catch (UnavailableContainerException e) {
            throw new XletStateChangeException(e.getMessage());

        } catch (Exception e){
            throw new XletStateChangeException(e.getMessage());
        }
        
        try {
			cardManager = new CardManager();
						
		} catch (InitializeException e) {
			msgLabel.setText(new Text("ERR_INIT"));
			e.printStackTrace();
		}
		
		if (listener == null) {
			setSlot(null);
			setProtocol(-1);
			setAtr(null);
			setResponse(null);
			
			setMessage(new Text("MSG_SETCARD"));
			
			CardEventListener listener = new CardEventListener() {
				
				/**
				 * The process followed after a card is inserted to the card reader. 
				 * 1.obtains a Card object.
				 * 2.obtains a mutex lock of the card.
				 * 3.uses Card.Info object.
				 *   - obtains slot information
				 *   - obtains protocol information
				 *   - obtains atr of the card
				 * 4.uses Card.IO object (if request is not null).
				 *   - sends APDU request to the card and receives APDU response
				 * 5.releases the mutex lock of the card.
				 * 6.sets LED state.
				 * 
				 * カードが、認識された時の処理
				 * 1.カードを取得する。
				 * 2.カードをロックする。
				 * 3.Card.Infoオブジェクトを使用する。
				 *   - スロット情報の取得
				 *   - プロトコル情報の取得
				 *   - カードATRの取得
				 * 4.Card.IOオブジェクトを使用する。
				 *   - APDUリクエストをカードに送信、APDUレスポンスをカードから受信
				 * 5.カードのロックを解除する。
				 * 6.LEDを設定する。
				 */
				public void inserted(CardEvent event) {
					Card card = event.getSlot().getCard();
					
					try {
						try {
							card.lock();
							
							Info cardInfo = card.getInfo();
							
							Slot slot = cardInfo.getSlot();
							setSlot(slot.getId());
															
							int protocol = cardInfo.getProtocol();
							setProtocol(protocol);
							
							byte[] atr = cardInfo.getAtr();
							setAtr(atr);
							
							IO io = card.getIO();
							
							if (request != null) {
								APDU res = io.transmit(new APDU(request));
								setResponse(res.toBytes());
							}
							
							setMessage(new Text("MSG_SUCCESS_INSERTED"));
							
						} finally {
							card.unlock();
						}
						
					} catch (CardAccessException e) {
						setMessage(new Text("ERR_CARDACCESS"));
						e.printStackTrace();
						
					} catch (SlotAccessException e) {
						setMessage(new Text("ERR_SLOTACCESS"));
						e.printStackTrace();
						
					} catch (Throwable t) {
						setMessage(new Text("ERR_THROWABLE"));
						t.printStackTrace();
					}
					
					setLED(Type.START, Operation.GRN_BLINK);
				}
				
				/**
				 * The process followed after a card is removed from the card reader.
				 * 1.displays a message that indicates the removal of a card.
				 * 2.sets LED state.
				 * 
				 * カードが、外された時の処理
				 * 1.カードが外された旨を表示する。
				 * 2.LEDを設定する。
				 */
				public void removed(CardEvent event) {
					setSlot(null);
					setProtocol(-1);
					setAtr(null);
					setResponse(null);
					
					setMessage(new Text("MSG_SUCCESS_REMOVED"));
					
					setLED(Type.START, Operation.RED_BLINK);
				}
				
			};
			
			cardManager.addCardListener(listener);
			
			this.listener = listener;
			
			setLED(Type.START, Operation.RED_BLINK);
		}
	}

	/**
	 * Activation
	 * 
	 * 活性化。
	 */
	synchronized public void startXlet() throws XletStateChangeException {
	}
	
	/**
	 * Stop.
	 * 
	 * 停止。
	 */
	synchronized public void pauseXlet() {
	}

	/**
	 * End process.
	 * This process is similar to the process followed after the clear/stop key is pressed.
	 * 
	 * 終了処理。
	 * クリアストップキー押された時と同様の処理を行う。
	 * 
	 */
	synchronized public void destroyXlet(boolean destroy) throws XletStateChangeException {
		if (listener != null) {
			cardManager.removeCardListener(listener);
			listener = null;
			
			cardManager.abort();
			
			setMessage(new Text("ERR_ABORT"));
			
			setLED(Type.START, Operation.GRN_ON);
		}
	}
	
	/**
	 * GUI creation.
	 * This is called from the initXlet method, and creates GUI on the operation panel.
	 * 1.creates the title window.
	 * 2.creates the title label.
	 * 3.creates the message label.
	 * 4.creates the slot label.
	 * 5.creates the protocol label.
	 * 6.creates the ATR label.
	 * 7.creates the APDU request label.
	 * 8.creates the button to display the soft keyboard.
	 * 9.implements the soft keyboard.
	 * 10.creates the APDU response label.
	 * 11.registers a KeyListener to the root frame for processing KeyEvents. 
	 * 
	 * GUI作成。
	 * initXletメソッドから呼び出され、オペパネにＧＵＩを作成します。
	 * 1.タイトル表示ウィンドウを作成する。
	 * 2.タイトル表示ラベルを作成する。
	 * 3.メッセージ表示ラベルを作成する。
	 * 4.スロット表示ラベルを作成する。
	 * 5.プロトコル表示ラベルを作成する。
	 * 6.ATR表示ラベルを作成する。
	 * 7.リクエスト表示ラベルを作成する。
	 * 8.リクエスト入力キーボードの表示ボタンを作成する。
	 * 9.リクエスト入力キーボードを実装する。
	 * 10.レスポンス表示ラベルを作成する。
	 * 11.キーリスナーを登録して、キーイベントを処理する。
	 */
	private void createGUI() {
		Window titleWindow = new Window(frame);
		titleWindow.setAttributeType(AttributeType.INVERSE);
		titleWindow.setLocation(0, 0);
		titleWindow.setSize(frame.getWidth(), 32);

		Label titleLabel = new Label(new Text("APP_TITLE"));  
		titleLabel.setFont(Font.F16);
		titleLabel.setAttributeType(AttributeType.INVERSE);
		titleLabel.setLocation(0, 0);
		titleLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(titleLabel);
		
		msgLabel = new Label(new Text("MSG_PUSHSTART"));  
		msgLabel.setFont(Font.F16);
		msgLabel.setAttributeType(AttributeType.INVERSE);
		msgLabel.setLocation(0, 16);
		msgLabel.setSize(titleWindow.getWidth(), 16);
		titleWindow.add(msgLabel);

		slotLabel = new Label();
		slotLabel.setSize(titleWindow.getWidth()-200, 16);
		slotLabel.setLocation(
				100 ,
				((frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 5) * 1);
		frame.add(slotLabel);
		setSlot(null);
		
		protocolLabel = new Label();
		protocolLabel.setSize(titleWindow.getWidth()-200, 16);
		protocolLabel.setLocation(
				100 ,
				((frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 5) * 2);
		frame.add(protocolLabel);
		setProtocol(-1);
		
		atrLabel = new Label();
		atrLabel.setSize(titleWindow.getWidth()-200, 16);
		atrLabel.setLocation(
				100 ,
				((frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 5) * 3);
		frame.add(atrLabel);
		setAtr(null);
		
		requestLabel = new Label();
		requestLabel.setSize(titleWindow.getWidth()-200, 16);
		requestLabel.setLocation(
				100 ,
				((frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 5) * 4);
		frame.add(requestLabel);
		setRequest(null);
		
		keyButton = new Button(new Text("KEYBUTTON"));
		keyButton.setBounds(40, 
				((frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 5) * 4,
				50, 20);
		keyButton.addActionListener(this);
		frame.add(keyButton);
		
		keyboard = new SoftKeyboard(frame, new Text("SOFTKEYBOARD"));
		keyboard.setLanguage(Language.en);		
		keyboard.addKeyListener(new KeyAdapter() {
	         public void keyPressed(KeyEvent event) {
	             String str = null;
	             SoftKeyboard obj = (SoftKeyboard)event.getSource();
	             str = obj.getStrdata();
	             
	             if ((str == null) || (str.length()%2 != 0)) {
	            	 request = null;
	            	 setRequest(request);
	            	 return;
	             }
	             
	             request = new byte[str.length()/2];
	             try {
	            	 for (int offset = 0; offset <= str.length() - 2; offset += 2){
	            		 String hexByte = str.substring(offset, offset+2);
	            		 request[offset/2] = (byte)Integer.parseInt(hexByte, 0x10);
	            		}
	            	 setRequest(request);
	            	 
	             } catch (NumberFormatException e) {
	            	 request = null;
	            	 setRequest(request);
	             }
	         }
	     });
		
		responseLabel = new Label();
		responseLabel.setSize(titleWindow.getWidth()-200, 16);
		responseLabel.setLocation(
				100 ,
				((frame.getHeight() - titleWindow.getHeight() - msgLabel.getHeight()) / 5) * 5);
		frame.add(responseLabel);
		setResponse(null);
	}
	
	/**
	 * Implementation of the action listener.
	 * Displays the soft keyboard.
	 * 
	 * アクションリスナの実装
	 * ソフトキーボードを表示する。
	 */
	public void actionPerformed(ActionEvent event) {
		Button src = (Button) event.getSource();
		
		if (src.equals(keyButton)) {
			try {
				keyboard.show();
				
			} catch (SoftKeyboardOpenException e) {
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * Displays the given slot informmation on the slot label.
	 * 
	 * スロット情報を、スロット表示ラベルに表示する。
	 */
	private void setSlot(String slot) {
		final String replace = "NUMBER";
		Text text = new Text("SLOT_LABEL");
		
		if (slot == null) {
			text.addReplace(replace, "");
		} else { 
			text.addReplace(replace, slot);
		}
		slotLabel.setText(text);
		slotLabel.repaint();
	}
	
	/**
	 * Displays the given protocol information on the protocol label.
	 * 
	 * プロトコル情報を、プロトコル表示ラベルに表示する。
	 */
	private void setProtocol(int protocol) {
		final String replace = "NUMBER";
		Text text = new Text("PROTOCOL_LABEL");
		
		if (protocol == -1) {
			text.addReplace(replace, "");
		} else { 
			text.addReplace(replace, Integer.toString(protocol));
		}
		protocolLabel.setText(text);
		protocolLabel.repaint();
	}
	
	/**
	 * Displays the given ATR information on the ATR label.
	 * 
	 * ATR情報を、ATR表示ラベルに表示する。
	 */
	private void setAtr(byte[] atr) {
		final String replace = "NUMBER";
		Text text = new Text("ATR_LABEL");
		
		if (atr == null) {
			text.addReplace(replace, "");
		} else { 
			String s="";
			int n = 0;
			for (int i = 0; i < atr.length; i++) {
				n = atr[i];
				n &=0x000000ff;
				if(n<=0x0000000f){
					s += "0";
				}
				s += Integer.toHexString(n)+" ";
			}
			text.addReplace(replace, s);
		}
		atrLabel.setText(text);
		atrLabel.repaint();
	}
	
	/**
	 * Displays the given APDU request on the request label.
	 * 
	 * APDUリクエストを、リクエスト表示ラベルに表示する。
	 */
	private void setRequest(byte[] request) {
		final String replace = "NUMBER";
		Text text = new Text("REQUEST_LABEL");
		
		if (request == null) {
			text.addReplace(replace, "");
		} else { 
			String s="";
			int n = 0;
			for (int i = 0; i < request.length; i++) {
				n = request[i];
				n &=0x000000ff;
				if(n<=0x0000000f){
					s += "0";
				}
				s += Integer.toHexString(n)+" ";
			}
			text.addReplace(replace, s);
		}
		requestLabel.setText(text);
		requestLabel.repaint();
	}
	
	/**
	 * Displays the given APDU response on the response label.
	 * 
	 * APDUレスポンスを、レスポンス表示ラベルに表示する。
	 */
	private void setResponse(byte[] response) {
		final String replace = "NUMBER";
		Text text = new Text("RESPONSE_LABEL");
		
		if (response == null) {
			text.addReplace(replace, "");
		} else { 
			String s="";
			int n = 0;
			for (int i = 0; i < response.length; i++) {
				n = response[i];
				n &=0x000000ff;
				if(n<=0x0000000f){
					s += "0";
				}
				s += Integer.toHexString(n)+" ";
			}
			text.addReplace(replace, s);
		}
		responseLabel.setText(text);
		responseLabel.repaint();
	}
	
	/**
	 * Displays the given message on the message label.
	 * 
	 * メッセージを、メッセージ表示ラベルに表示する。
	 */
	private void setMessage(Text message) {
		msgLabel.setText(message);
		msgLabel.repaint();
	}
	
	/**
	 * Activates LED with the given Type and the given Operation.
	 * 
	 * LEDを動作させる。
	 */
	private void setLED(Type type, Operation ope) {
		try {
			new LED(frame, type, ope).blink();
		} catch (LEDExecutionException e) {
			e.printStackTrace();
		}
	}
}
